<?php
/**
 * @version		$Id: user.php 345 2009-02-27 19:35:17Z robs $
 * @package		JXtended.Libraries
 * @subpackage	Database
 * @copyright	Copyright (C) 2008 - 2009 JXtended, LLC. All rights reserved.
 * @license		GNU General Public License <http://www.gnu.org/copyleft/gpl.html>
 * @link		http://jxtended.com
 */

defined('JPATH_BASE') or die;

require_once JPATH_LIBRARIES.DS.'joomla'.DS.'database'.DS.'table'.DS.'user.php';

/**
 * User table class for JXtended Libraries.
 *
 * @package		JXtended.Libraries
 * @subpackage	Database
 * @version		1.0
 */
class JXTableUser extends JTableUser
{
	/**
	 * Associative array of user group ids => names.
	 *
	 * @access	public
	 * @since	1.0
	 * @var		array
	 */
	var $groups;

	/**
	 * Method to bind the user, user groups, and any other necessary data.
	 *
	 * @access	public
	 * @param	array		$array		The data to bind.
	 * @param	mixed		$ignore		An array or space separated list of fields to ignore.
	 * @return	boolean		True on success, false on failure.
	 * @since	1.0
	 */
	function bind($array, $ignore = '')
	{
		// Attempt to bind the data.
		$return = parent::bind($array, $ignore);

		// Load the real group data based on the bound ids.
		if ($return && !empty($this->groups))
		{
			// Set the group ids.
			$this->groups = array_keys($this->groups);
			JArrayHelper::toInteger($this->groups);
			$this->groups = array_fill_keys(array_values($this->groups), null);

			// Get the titles for the user groups.
			$this->_db->setQuery(
				'SELECT `id`, `title`' .
				' FROM `#__usergroups`' .
				' WHERE `id` = '.implode(' OR `id` = ', array_keys($this->groups))
			);
			$results = $this->_db->loadObjectList();

			// Check for a database error.
			if ($this->_db->getErrorNum()) {
				$this->setError($this->_db->getErrorMsg());
				return false;
			}

			// Set the titles for the user groups.
			for ($i = 0, $n = count($results); $i < $n; $i++) {
				$this->groups[$results[$i]->id] = $results[$i]->title;
			}
		}

		return $return;
	}

	/**
	 * Method to delete a user, user groups, and any other necessary
	 * data from the database.
	 *
	 * @access	public
	 * @param	integer		$id		An optional user id.
	 * @return	boolean		True on success, false on failure.
	 * @since	1.0
	 */
	function delete($id = null)
	{
		// Attempt to delete the user.
		$return = parent::delete($id);

		// Delete the group maps if the user data was deleted.
		if ($return)
		{
			// Delete the user group maps.
			$this->_db->setQuery(
				'DELETE FROM `#__user_usergroup_map`' .
				' WHERE `user_id` = '.(int)$this->id
			);
			$this->_db->query();

			// Check for a database error.
			if ($this->_db->getErrorNum()) {
				$this->setError($this->_db->getErrorMsg());
				return false;
			}
		}

		return $return;
	}

	/**
	 * Method to load a user, user groups, and any other necessary data
	 * from the database so that it can be bound to the user object.
	 *
	 * @access	public
	 * @param	integer		$id		An optional user id.
	 * @return	boolean		True on success, false on failure.
	 * @since	1.0
	 */
	function load($id = null)
	{
		// Get the id to load.
		if ($id !== null) {
			$this->id = $id;
		} else {
			$id = $this->id;
		}

		// Check for a valid id to load.
		if ($id === null) {
			return false;
		}

		// Reset the table.
		$this->reset();

		// Load the user data.
		$this->_db->setQuery(
			'SELECT *' .
			' FROM #__users' .
			' WHERE id = '.(int)$id
		);
		$data = (array)$this->_db->loadAssoc();

		// Check for an error message.
		if ($this->_db->getErrorNum()) {
			$this->setError($this->_db->getErrorMsg());
			return false;
		}

		// Bind the data to the table.
		$return = $this->bind($data);

		if ($return !== false)
		{
			// Load the user groups.
			$this->_db->setQuery(
				'SELECT g.id, g.title' .
				' FROM #__usergroups AS g' .
				' JOIN #__user_usergroup_map AS m ON m.group_id = g.id' .
				' WHERE m.user_id = '.(int)$id
			);
			$result = $this->_db->loadObjectList();
			$groups	= array();

			// Check for an error message.
			if ($this->_db->getErrorNum()) {
				$this->setError($this->_db->getErrorMsg());
				return false;
			}

			// Create an array of groups.
			for ($i = 0, $n = count($result); $i < $n; $i++) {
				$groups[$result[$i]->id] = $result[$i]->title;
			}

			// Add the groups to the user data.
			$this->groups = $groups;
		}

		return $return;
	}

	/**
	 * Method to store a user, user groups, and any other necessary
	 * data to the database.
	 *
	 * @access	public
	 * @param	boolean		$updateNulls	Toggle whether null values should be updated.
	 * @return	boolean		True on success, false on failure.
	 * @since	1.0
	 */
	function store($updateNulls = false)
	{
		// Attempt to store the user data.
		$return = parent::store($updateNulls);

		// Store the group data if the user data was saved.
		if ($return && is_array($this->groups) && count($this->groups))
		{
			// Delete the old user group maps.
			$this->_db->setQuery(
				'DELETE FROM `#__user_usergroup_map`' .
				' WHERE `user_id` = '.(int)$this->id
			);
			$this->_db->query();

			// Check for a database error.
			if ($this->_db->getErrorNum()) {
				$this->setError($this->_db->getErrorMsg());
				return false;
			}

			// Set the new user group maps.
			$this->_db->setQuery(
				'INSERT INTO `#__user_usergroup_map` (`user_id`, `group_id`)' .
				' VALUES ('.$this->id.', '.implode('), ('.$this->id.', ', array_keys($this->groups)).')'
			);
			$this->_db->query();

			// Check for a database error.
			if ($this->_db->getErrorNum()) {
				$this->setError($this->_db->getErrorMsg());
				return false;
			}
		}

		return $return;
	}
}