<?php
/**
 * @version		$Id: aclreferences.php 154 2008-11-29 21:55:24Z louis $
 * @package		JXtended.Libraries
 * @subpackage	Acl
 * @copyright	Copyright (C) 2008 JXtended, LLC. All rights reserved.
 * @license		GNU General Public License
 */

defined('JPATH_BASE') or die;

/**
 * Object to work generically with referenced data
 *
 * @package		JXtended.Libraries
 * @subpackage	Acl
 */
class JxAclReferences
{
	/**
	 * The references ACLs
	 * @var array
	 * @protected
	 */
	var $_acls = array();
	/**
	 * The references ACOs
	 * @var array
	 * @protected
	 */
	var $_acos = array();
	/**
	 * The references AROs
	 * @var array
	 * @protected
	 */
	var $_aros = array();
	/**
	 * The references AXOs
	 * @var array
	 * @protected
	 */
	var $_axos = array();
	/**
	 * The references ARO Groups
	 * @var array
	 * @protected
	 */
	var $_aro_groups = array('__default' => array());
	/**
	 * The references AXO Groups
	 * @var array
	 * @protected
	 */
	var $_axo_groups = array('__default' => array());

	/**
	 * Generic add method
	 *
	 * @param	string $field		The field to operate on
	 * @param	string $section		The section
	 * @param	mixed $value		An array or a string
	 *
	 * @access	protected
	 */
	function _add($field, $section, $value)
	{
		$f = &$this->$field;
		if (!isset($f[$section])) {
			$f[$section] = array();
		}
		if (is_array($value)) {
			$f[$section] = array_merge($f[$section], $value);
		}
		else {
			$f[$section][] = $value;
		}
	}

	/**
	 * Add the an ACL
	 *
	 * @param	string $section		The section
	 * @param	mixed $value		An array or a string
	 *
	 * @access	public
	 */
	function addAcl($section, $value)
	{
		$this->_add('_acls', $section, $value);
	}

	/**
	 * Add the an ACO
	 *
	 * @param	string $section		The section
	 * @param	mixed $value		An array or a string
	 *
	 * @access	public
	 */
	function addAco($section, $value)
	{
		$this->_add('_acos', $section, $value);
	}

	/**
	 * Add the an ARO
	 *
	 * @param	string $section		The section
	 * @param	mixed $value		An array or a string
	 *
	 * @access	public
	 */
	function addAro($section, $value)
	{
		$this->_add('_aros', $section, $value);
	}

	/**
	 * Add the an AXO
	 *
	 * @param	string $section		The section
	 * @param	mixed $value		An array or a string
	 *
	 * @access	public
	 */
	function addAxo($section, $value)
	{
		$this->_add('_axos', $section, $value);
	}

	/**
	 * Add the an ARO Group
	 *
	 * @param	string $section		The section
	 * @param	mixed $value		An array or a string
	 *
	 * @access	public
	 */
	function addAroGroup($value)
	{
		$this->_add('_aro_groups', '__default', $value);
	}

	/**
	 * Add the an AXO Group
	 *
	 * @param	string $section		The section
	 * @param	mixed $value		An array or a string
	 *
	 * @access	public
	 */
	function addAxoGroup($value)
	{
		$this->_add('_axo_groups', '__default', $value);
	}

	/**
	 * Bind an input array to this class
	 *
	 * array(
	 * 	'acls' => array( 'section' => array( 'value' ) ),
	 * 	'acos' => array( 'section' => array( 'value' ) ),
	 * 	'aros' => array( 'section' => array( 'value' ) ),
	 * 	'axos' => array( 'section' => array( 'value' ) ),
	 * 	'aro_groups' => array( 'value' ),
	 * 	'axo_groups' => array( 'value' ),
	 * )
	 *
	 * This is mainly used by the Save ACL methods
	 *
	 * @param	array $input
	 */
	function bind($input)
	{
		if (isset($input['acls']) && is_array($input['acls']))
		{
			foreach ($input['acls'] as $section => $value) {
				$this->addAco($section, $value);
			}
		}

		if (isset($input['acos']) && is_array($input['acos']))
		{
			foreach ($input['acos'] as $section => $value) {
				$this->addAco($section, $value);
			}
		}

		if (isset($input['aros']) && is_array($input['aros']))
		{
			foreach ($input['aros'] as $section => $value) {
				$this->addAro($section, $value);
			}
		}

		if (isset($input['axos']) && is_array($input['axos']))
		{
			foreach ($input['axos'] as $section => $value) {
				$this->addAxo($section, $value);
			}
		}

		if (isset($input['aro_groups']) && is_array($input['aro_groups']))
		{
			$this->addAroGroup($input['aro_groups']);
		}

		if (isset($input['axo_groups']) && is_array($input['axo_groups']))
		{
			$this->addAxoGroup($input['axo_groups']);
		}

		return true;
	}

	/**
	 * Get the referenced ACLs
	 *
	 * @param	string $field		The field to operate on
	 * @param	mixed $section		True to return array with sections, string for a particular section, otherwise a concatenated array is returned
	 *
	 * @return	array
	 * @access	public
	 */
	function _get($field, $section = null)
	{
		$result = array();
		$f = &$this->$field;

		if ($section === true) {
			return $f;
		}
		else if (empty($section)) {
			foreach ($f as $section => $value) {
				$result = array_merge($result, $value);
			}
		}
		else if (isset($f[$section])) {
			return $f[$section];
		}
		return $result;
	}

	/**
	 * Get the referenced ACLs
	 *
	 * @param	mixed $section		True to return array with sections, string for a particular section, otherwise a concatenated array if returned
	 *
	 * @return	array
	 * @access	public
	 */
	function getAcls($section = null)
	{
		return $this->_get('_acls', $section);
	}

	/**
	 * Get the referenced ACOs
	 *
	 * @param	mixed $section		True to return array with sections, string for a particular section, otherwise a concatenated array if returned
	 *
	 * @return	array
	 * @access	public
	 */
	function getAcos($section = null)
	{
		return $this->_get('_acos', $section);
	}

	/**
	 * Get the referenced AROs
	 *
	 * @param	mixed $section		True to return array with sections, string for a particular section, otherwise a concatenated array if returned
	 *
	 * @return	array
	 * @access	public
	 */
	function getAros($section = null)
	{
		return $this->_get('_aros', $section);
	}

	/**
	 * Get the referenced AXOs
	 *
	 * @param	mixed $section		True to return array with sections, string for a particular section, otherwise a concatenated array if returned
	 *
	 * @return	array
	 * @access	public
	 */
	function getAxos($section = null)
	{
		return $this->_get('_axos', $section);
	}

	/**
	 * Get the referenced ARO Groups
	 *
	 * @return	array
	 * @access	public
	 */
	function getAroGroups()
	{
		return $this->_get('_aro_groups', '__default');
	}

	/**
	 * Get the referenced AXO Groups
	 *
	 * @return	array
	 * @access	public
	 */
	function getAxoGroups()
	{
		return $this->_get('_axo_groups', '__default');
	}

	/**
	 * Gets a signature so that changes can be detected
	 *
	 * @return	string
	 * @access	public
	 */
	function getSignature()
	{
		return md5(
			array(
				$this->_acls,
				$this->_acos,
				$this->_aros,
				$this->_axos,
				$this->_aro_groups,
				$this->_axo_groups
			)
		);
	}

	/**
	 * A utility method to quickly check if there are no references
	 *
	 * @return	booelan
	 * @access	public
	 */
	function isEmpty()
	{
		return empty($this->_acls)
			& empty($this->_acos)
			& empty($this->_aros)
			& empty($this->_axos)
			& empty($this->_aro_groups['__default'])
			& empty($this->_axo_groups['__default']);
	}
}
