<?php
/**
 * @version		$Id: user.php 254 2009-01-07 23:57:23Z louis $
 * @package		JXtended.Libraries
 * @subpackage	User
 * @copyright	Copyright (C) 2005 - 2009 Open Source Matters, Inc. All rights reserved.
 * @copyright	Copyright (C) 2008 - 2009 JXtended, LLC. All rights reserved.
 * @license		GNU General Public License <http://www.gnu.org/copyleft/gpl.html>
 * @link		http://jxtended.com
 */

defined('JPATH_BASE') or die;

/**
 * User class.  Handles all application interaction with a user
 *
 * @package 	Joomla.Framework
 * @subpackage	User
 * @version		1.6
 */
class JUser extends JObject
{
	/**
	 * The user id.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		integer
	 */
	public $id;

	/**
	 * The full name of the user.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		string
	 */
	public $name;

	/**
	 * The login name of the user.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		string
	 */
	public $username;

	/**
	 * The email address of the user.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		string
	 */
	public $email;

	/**
	 * The password for the user.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		string
	 */
	public $password;

	/**
	 * The clear text password for the user.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		string
	 */
	public $password_clear;

	/**
	 * Flag to indicate the user is activated.
	 *
	 * @access	public
	 * @since	1.6
	 * @var		integer
	 */
	public $activated;

	/**
	 * Flag to indicate the user is enabled.
	 *
	 * @access	public
	 * @since	1.6
	 * @var		integer
	 */
	public $enabled;

	/**
	 * Flag to indicate the user class.
	 *
	 * @access	public
	 * @since	1.6
	 * @var		integer
	 */
	public $class;

	/**
	 * General authorization token for the user.
	 *
	 * @access	public
	 * @since	1.6
	 * @var		string
	 */
	public $token;

	/**
	 * Flag to indicate the user's e-mail preferences.
	 *
	 * @access	public
	 * @since	1.6
	 * @var		integer
	 */
	public $send_email;

	/**
	 * User register data as JDate object.
	 *
	 * @access	public
	 * @since	1.6
	 * @var		string
	 */
	public $register_date;

	/**
	 * User last visit date as JDate object.
	 *
	 * @access	public
	 * @since	1.6
	 * @var		string
	 */
	public $last_visit_date;

	/**
	 * User parameters as JParameter object.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		string
	 */
	public $params;

	/**
	 * User access level.
	 *
	 * Note: Deprecated in 1.6.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		integer
	 */
	public $aid;

	/**
	 * Flag to indicated the user is a guest.
	 *
	 * @access	public
	 * @since	1.5
	 * @var		boolean
	 */
	public $guest;

	/**
	 * Associative array of user group ids => names.
	 *
	 * @access	protected
	 * @since	1.6
	 * @var		array
	 */
	protected $_groups;

	/**
	 * Method to add the user to a group.
	 *
	 * @access	public
	 * @param	integer		$id		The id of the group to add the user to.
	 * @since	1.6
	 */
	public function addGroup($id)
	{
		if (!array_key_exists($id, $this->_groups)) {
			$this->_groups[$id] = null;
		}
	}

	/**
	 * Method to get the groups a user is in.
	 *
	 * @access	public
	 * @return	array		An asssociatve array of user group ids => names.
	 * @since	1.6
	 */
	public function getGroups()
	{
		// Check if we need to look up group names.
		if (in_array(null, $this->_groups, true))
		{
			// Get the group ids.
			$ids = array_keys($this->_groups);
			JArrayHelper::toInteger($ids);

			// Load the group names from the database by id.
			$query	= 'SELECT id, title FROM #__usergroups'
					. ' WHERE id = '.implode(' OR id = ', $ids);
			$db	= &JFactory::getDBO();
			$db->setQuery($query);
			$results = $db->loadObjectList('id');

			// Check for a database error.
			if ($db->getErrorNum()) {
				$this->setError($db->getErrorMsg());
				return false;
			}

			// Set the group names.
			foreach ($this->_groups as $id => $title) {
				if (array_key_exists($id, $results)) {
					$this->_groups[$id] = $results[$id]->title;
				}
			}
		}

		return $this->_groups;
	}

	/**
	 * Method to set the groups a user is in.
	 *
	 * @access	public
	 * @param	array		$groups		An array of group ids for the user.
	 * @return
	 */
	public function setGroups($groups)
	{
		JArrayHelper::toInteger($groups);
		$this->_groups = array_fill_keys(array_values($groups), null);
	}

	/**
	 * Method to remove the user from a group.
	 *
	 * @access	public
	 * @param	integer		$id		The id of the group to remove the user from.
	 * @since	1.6
	 */
	public function removeGroup($id)
	{
		// Remove the user from the group.
		if (array_key_exists($id, $this->_groups)) {
			unset($this->_groups[$id]);
		}
	}

	function authorize()
	{
		// Method to authorize a user for an action[ on an asset].
	}

	/**
	 * Method to bind an associative array of data to a user object.
	 *
	 * @access 	public
	 * @param 	array 		$data		The associative array to bind to the object.
	 * @return 	boolean 	True on success
	 * @since	1.5
	 */
	function bind(&$data)
	{
		// If the user is new or the password has been changed, create a new password hash.
		if (empty($this->id) || (!empty($this->id) && !empty($data['password'])))
		{
			jimport('joomla.user.helper');

			// If the user is new and no password is provided, create a random one.
			if (empty($this->id) && empty($data['password'])) {
				$data['password'] = JUserHelper::genRandomPassword();
			}

			// Set aside a copy of the clear text password.
			$this->password_clear = JArrayHelper::getValue($data, 'password', '', 'string');

			// Create the password hash with salt.
			$salt	= JUserHelper::genRandomPassword(32);
			$crypt	= JUserHelper::getCryptedPassword($data['password'], $salt);
			$data['password'] = $crypt.':'.$salt;
		}
		else
		{
			// Password has not changed, use the current password.
			$data['password'] = $this->password;
		}

		// Bind the params if present.
		if (array_key_exists('params', $data)) {
			$this->params->bind($data['params']);
		}

		// Bind the data.
		if (!$this->setProperties($data)) {
			$this->setError("Unable to bind array to user object");
			return false;
		}

		// Make sure its an integer
		$this->id = (int)$this->id;

		return true;
	}

	function delete()
	{
		// Method to delete a user.
	}

	function load($id)
	{
		// Get a user table object.
		$table = &JTable::getInstance('User', 'JTable');

		// Fire the onBeforeLoadUser event.
		JPluginHelper::importPlugin('user');
		$dispatcher =& JDispatcher::getInstance();
		$dispatcher->trigger('onBeforeLoadUser', array($id, &$this));

		// Load the user data.
		if(!$table->load($id)) {
			$this->setError($table->getError());
			return false;
		}

		// Bind the user data.
		$this->setProperties($table->getProperties());

		// Convert user data strings to objects.
		$this->params = new JParameter($this->params);
		$this->register_date = &JFactory::getDate($this->register_date);
		$this->last_visit_date = &JFactory::getDate($this->last_visit_date);

		// Load the user groups.
		$db = &JFactory::getDBO();
		$db->setQuery(
			'SELECT g.id, g.title' .
			'FROM #__usergroups AS g' .
			' JOIN #__user_usergroup_map AS m ON m.group_id = g.id' .
			' WHERE m.user_id = '.(int)$id
		);
		$groups = $db->loadObjectList('id');

		// Set the user groups.
		foreach ($groups as $gk => $gv) {
			$this->_groups[$gk] = $gv->title;
		}

		// Fire the onAftereLoadUser event.
		$dispatcher->trigger('onAfterLoadUser', array($id, &$this));

		return true;
	}

	/**
	 * Method to save the user to the database.
	 *
	 * @access 	public
	 * @param 	boolean		$updateOnly		Save the object only if not a new user
	 * @return 	boolean 	True on success, false on failure.
	 * @since	1.5
	 */
	function store($updateOnly = false)
	{
		// Get a user table object.
		$table = &JTable::getInstance('User', 'JTable');

		// Convert user data objects to strings.
		$data = $this->getProperties();
		$data['params']	= $data['params']->toString();
		$data['register_date'] = $data['register_date']->toMySQL();
		$data['last_visit_date'] = $data['last_visit_date']->toMySQL();

		// Bind the user data to the table.
		$table->bind($data);

		// Check and store the object.
		if (!$table->check()) {
			$this->setError($table->getError());
			return false;
		}

		/*
		 * At some point, we need to check if the user is trying to create a user
		 * that is at their same access level and whether or not they are allowed
		 * to do that. This code used to be right here but was removed because it
		 * had hard coded group ids.
		 */

		// Set a flag if the user is new.
		$isNew = !$this->id;

		// If we aren't allowed to create new users return.
		if ($isNew && $updateOnly) {
			return true;
		}

		// Get the old user data.
		$old = new JUser($this->id);

		// Fire the onBeforeStoreUser event.
		JPluginHelper::importPlugin('user');
		$dispatcher =& JDispatcher::getInstance();
		$dispatcher->trigger('onBeforeStoreUser', array(&$old, $isNew));

		/*
		 * Store the user data and the user groups in the database.
		 */
		try
		{
			$db = &JFactory::getDBO();

			// Store the user data in the database.
			$return = $table->store();

			// Check for a database error.
			if (!$return) {
				throw new JException($table->getError());
			}

			// Set the id for the JUser object in case we created a new user.
			if (empty($this->id)) {
				$this->id = $table->get('id');
			}

			// Delete the old user group maps.
			$db->setQuery(
				'DELETE FROM `#__user_usergroup_map`' .
				' WHERE `user_id` = '.(int)$this->id
			);
			$return = $db->query();

			// Check for a database error.
			if ($db->getErrorNum()) {
				throw new JException($db->getErrorMsg());
			}

			// Set the new user group maps.
			$db->setQuery(
				'INSERT INTO #__user_usergroup_map (`user_id`, `group_id`)' .
				' VALUES ('.$this->id.', '.implode('), ('.$this->id.', ', array_keys($this->_groups)).')'
			);
			$return = $db->query();

			// Check for a database error.
			if ($db->getErrorNum()) {
				throw new JException($db->getErrorMsg());
			}
		}
		catch (JException $e)
		{
			// Set the error message.
			$this->setError($e->getMessage());
		}

		// Fire the onAftereStoreUser event.
		$dispatcher->trigger('onAfterStoreUser', array(&$this, $isNew, $return, $this->getError()));

		return $return;
	}
}